TITLE_IMAGE = new Image();
TITLE_IMAGE.src = "HexLife_title.png";
INSTRUCTIONS_IMAGE = new Image();
INSTRUCTIONS_IMAGE.src = "HexLife_instructions.png";

HEX_CELL_TYPES = {
    EMPTY : 0,
    RED : 1,
    RED_FOOD : 2,
    GREEN : 3,
    GREEN_FOOD : 4,
    BLUE : 5,
    BLUE_FOOD : 6,
    WALL : 7
}
HEX_CELL_TYPE_CHARS = ".RrGgBb*"
class HexLifeGrid {
    gridWidth;
    gridHeight;
    gridData;

    constructor(w, h) {
        this.gridWidth = w;
        this.gridHeight = h;
        this.gridData = [];
        for (let y = 0; y < h; ++y) {
            let row = [];
            for (let x = 0; x < w; ++x) {
                row[x] = 0;
            }
            this.gridData[y] = row;
        }
    }


    doesTileContain(x, y, tileSet) {
        let contains = false;
        let t = this.gridData[y][x];
        for (let i = 0; i < tileSet.length; ++i) {
            if (t === tileSet[i]) {
                contains = true;
                break;
            }
        }
        return contains;
    }

    loadFromString(s) {
        let x=0, y=0, index = 0;
        while (index < s.length) {
            let t = HEX_CELL_TYPE_CHARS.indexOf(s.charAt(index));
            if (t >= 0) {
                this.gridData[y][x] = t;
                ++x;
                if (x >= this.gridWidth) {
                    x=0;
                    ++y;
                }
            }
            ++index;
        }
    }

    clear() {
        for (let y = 0; y < this.gridWidth; ++y) {
            for (let x = 0; x < this.gridHeight; ++x) {
                this.gridData[y][x] = HEX_CELL_TYPES.EMPTY;
            }
        }
    }
    setCell(x, y, tile) {
        this.gridData[y][x] = tile;
    }

    getCell(x, y) {
        return this.gridData[y][x];
    }

    toString() {
        let s = "";
        for (let y = 0; y < this.gridWidth; ++y) {
            let r = ((y % 2) === 0) ? "" : " ";
            for (let x = 0; x < this.gridHeight; ++x) {
                r = r + "|" + HEX_CELL_TYPE_CHARS.charAt(this.gridData[y][x]);
            }
            s = s + r + "|\n";
        }
        return s;
    }

    toCondensedString() {
        let s = "";
        for (let y = 0; y < this.gridWidth; ++y) {
            for (let x = 0; x < this.gridHeight; ++x) {
                s = s + HEX_CELL_TYPE_CHARS.charAt(this.gridData[y][x]);
            }
            s = s + "\n";
        }
        return s;
    }

    validateAndCountTile(x, y, tile) {
        if ((x < 0) || (x >= this.gridWidth)) return 0;
        if ((y < 0) || (y >= this.gridHeight)) return 0;
        return (this.gridData[y][x] === tile) ? 1 : 0;
    }

    countSurroundingTiles(x, y, tile) {
        let adjust = (y % 2) === 0 ? -1 : 0;
        let count = this.validateAndCountTile(x + adjust, y-1, tile);
        count += this.validateAndCountTile(x + adjust + 1, y-1, tile);
        count += this.validateAndCountTile(x - 1, y, tile);
        count += this.validateAndCountTile(x + 1, y, tile);
        count += this.validateAndCountTile(x + adjust, y+1, tile);
        count += this.validateAndCountTile(x + adjust + 1, y+1, tile);
        return count;
    }

    countTileOccurrences(tile) {
        let count = 0;
        for (let y = 0; y < this.gridHeight; ++y) {
            for (let x = 0; x < this.gridWidth; ++x) {
                if (this.gridData[y][x] === tile)
                    ++count;
            }
        }
        return count;
    }

}

class HexLifeGenerator {
    gameGrid; workGrid; snapshot;

    constructor(w, h) {
        this.gameGrid = new HexLifeGrid(w, h);
        this.workGrid = new HexLifeGrid(w, h);
        this.snapshot = new HexLifeGrid(w, h);
    }

    placeRandomTiles(tile, count) {
        let x, y;
        for (let i = 0; i < count; ++i) {
            do {
                x = Math.floor(Math.random() * this.gameGrid.gridWidth);
                y = Math.floor(Math.random() * this.gameGrid.gridHeight);
            } while(this.gameGrid.doesTileContain(x,y,
                [HEX_CELL_TYPES.EMPTY])===false);
            this.gameGrid.setCell(x, y, tile);
        }
    }

    generateRandomGrid(red, green, blue) {
        this.placeRandomTiles(HEX_CELL_TYPES.RED, red);
        this.placeRandomTiles(HEX_CELL_TYPES.GREEN, green);
        this.placeRandomTiles(HEX_CELL_TYPES.BLUE, blue);
    }

    determineNewCellType(x, y) {
        let t = this.gameGrid.getCell(x, y);
        let counts = [];
        counts[HEX_CELL_TYPES.RED] = this.gameGrid.countSurroundingTiles(x, y, HEX_CELL_TYPES.RED);
        counts[HEX_CELL_TYPES.GREEN] = this.gameGrid.countSurroundingTiles(x, y, HEX_CELL_TYPES.GREEN);
        counts[HEX_CELL_TYPES.BLUE] = this.gameGrid.countSurroundingTiles(x, y, HEX_CELL_TYPES.BLUE);

        switch (t) {
            case HEX_CELL_TYPES.RED:
                if ( (counts[HEX_CELL_TYPES.BLUE] > 1) &&
                    (counts[HEX_CELL_TYPES.BLUE] > counts[HEX_CELL_TYPES.GREEN]) )//counts[HEX_CELL_TYPES.RED]) )
                    t = HEX_CELL_TYPES.BLUE;
                break;
            case HEX_CELL_TYPES.GREEN:
                if ( (counts[HEX_CELL_TYPES.RED] > 1) &&
                    (counts[HEX_CELL_TYPES.RED] > counts[HEX_CELL_TYPES.BLUE]) )// counts[HEX_CELL_TYPES.GREEN]) )
                    t = HEX_CELL_TYPES.RED;
                break;
            case HEX_CELL_TYPES.BLUE:
                if ( (counts[HEX_CELL_TYPES.GREEN] > 1) &&
                    (counts[HEX_CELL_TYPES.GREEN] > counts[HEX_CELL_TYPES.RED]) )//counts[HEX_CELL_TYPES.BLUE]) )
                    t = HEX_CELL_TYPES.GREEN;
                break;
            case HEX_CELL_TYPES.EMPTY:
                let index = HEX_CELL_TYPES.EMPTY;

                if ( (counts[HEX_CELL_TYPES.RED] > counts[HEX_CELL_TYPES.BLUE]) &&
                    (counts[HEX_CELL_TYPES.RED] > 1) )
                    index = HEX_CELL_TYPES.RED;
                if ( (counts[HEX_CELL_TYPES.GREEN] > counts[HEX_CELL_TYPES.RED]) &&
                    (counts[HEX_CELL_TYPES.GREEN] > 1) )
                    index = HEX_CELL_TYPES.GREEN;
                if ( (counts[HEX_CELL_TYPES.BLUE] > counts[HEX_CELL_TYPES.GREEN]) &&
                    (counts[HEX_CELL_TYPES.BLUE] > 1) )
                    index = HEX_CELL_TYPES.BLUE;
                t = index;

                break;
        }
        return t;
    }
    generateNextRound() {
        for (let y = 0; y < this.gameGrid.gridHeight; ++y) {
            for (let x = 0; x < this.gameGrid.gridWidth; ++x) {
                this.workGrid.setCell(x, y, this.determineNewCellType(x, y));
            }
        }
        let temp = this.gameGrid;
        this.gameGrid = this.workGrid;
        this.workGrid = temp;
    }

    createSnapshot() {
        for (let y = 0; y < this.gameGrid.gridHeight; ++y) {
            for (let x = 0; x < this.gameGrid.gridWidth; ++x) {
                this.snapshot.setCell(x, y, this.gameGrid.getCell(x, y));
            }
        }
    }

    restoreSnapshot() {
        for (let y = 0; y < this.gameGrid.gridHeight; ++y) {
            for (let x = 0; x < this.gameGrid.gridWidth; ++x) {
                this.gameGrid.setCell(x, y, this.snapshot.getCell(x, y));
            }
        }
    }
}

// GUI

class Rect {
    x = 0; y = 0;
    width = 1; height = 1;

    constructor(x, y, w, h) {
        this.x = x;
        this.y = y;
        this.width = w;
        this.height = h;
    }

    isInside(x, y) {
        return ( (x >= this.x) &&
            (x < (this.x + this.width)) &&
            (y >= this.y) &&
            (y < (this.y + this.height))
        );
    }

}

// GUIComponent

class GUIComponent {
    bounds;
    listeners = [];

    constructor(bounds) {
        this.bounds = bounds;
    }

    addListener(listener) {
        this.listeners.push(listener);
    }
    broadcast(message) {
        for (let l of this.listeners)
            l.guiMessage(this, message);
    }
    removeListener(listener) {
        let temp = [];
        for (let l of this.listeners)
            if (l !== listener)
                temp.push(l);
        this.listeners = temp;
    }

    mouseDown(x, y) { /*placeholder*/}
    mouseMove(x, y) { /*placeholder*/}
    mouseUp(x, y) { /*placeholder*/}
}

// Label

class Label extends GUIComponent {
    text;
    font = "24px fantasy";
    color = "#708";
    centered = true;

    constructor(text, bounds) {
        super(bounds);
        this.text = text;
        this.bounds = bounds;
    }

    render(ctx) {
        ctx.save();
        ctx.font = this.font;
        ctx.textBaseline = "top";
        ctx.fillStyle = this.color;
        if (this.centered) {
            ctx.textAlign = "center";
            ctx.fillText(this.text, this.bounds.x + this.bounds.width/2, this.bounds.y, this.bounds.width);
        } else {
            ctx.textAlign = "left";
            ctx.fillText(this.text, this.bounds.x, this.bounds.y, this.bounds.width);
        }
        ctx.restore();
    }
}

// --------------

class ImageLabel extends GUIComponent {
    image;
    clip;

    constructor(img, clip, bounds) {
        super(bounds);
        this.image = img;
        this.clip = clip;
    }

    render(ctx) {
        ctx.drawImage(this.image,
            this.clip.x, this.clip.y, this.clip.width, this.clip.height,
            this.bounds.x, this.bounds.y, this.bounds.width, this.bounds.height);
    }

}


// Button
class Button extends Label {
    borderColor = "#002";
    overColor = "#088";
    buttonColor = "#008";
    isDown = false;
    isOver = false;

    constructor(msg, bounds) {
        super(msg, bounds);
    }
    mouseDown(x, y) {
        this.isDown = this.bounds.isInside(x,y);
    }
    mouseMove(x,y) {
        this.isOver = this.bounds.isInside(x,y);
    }
    mouseUp(x,y) {
        if (
            (this.isDown) &&
            (this.bounds.isInside(x,y))
        )
            this.broadcast("clicked");
        this.isDown = false;
    };

    render(ctx) {
        let back = this.buttonColor;
        if ((this.isDown) || (this.isOver))
            back = this.overColor;
        ctx.save();
        ctx.fillStyle = back;
        ctx.strokeStyle = this.borderColor;
        ctx.strokeThickness = 3;
        ctx.strokeRect(this.bounds.x, this.bounds.y, this.bounds.width, this.bounds.height);
        ctx.fillRect(this.bounds.x, this.bounds.y, this.bounds.width, this.bounds.height);
        ctx.restore();
        super.render(ctx);
    }
}

//

HEX_CELL_COLORS = [
    "#EDF",
    "#F00", "#800",
    "#0F0", "#080",
    "#00F", "#008",
    "#102"
];

HEX_CELL_TRANSITIONS = [
    [0xEE, 0xDD, 0xFF],
    [0xFF, 0, 0], [0x88, 0, 0],
    [0, 0xFF, 0], [0, 0x88, 0],
    [0, 0, 0xFF], [0, 0, 0x88],
    [0x11, 0, 0x22]
];

class HexLifeGUIGrid extends GUIComponent {
    grid;
    transition = 1;
    transitionStep = .25;

    constructor(grid, bounds)
    {
        super(bounds);
        this.grid = grid;
    }
    drawHex(ctx, x, y, r) {
        let left = x - r;
        let right = x + r;
        let quarterY = y - r / 2;
        let threequarterY = y + r / 2;
        let top = y - r;
        let bottom = y + r;

        ctx.beginPath();
        ctx.moveTo(x, top);
        ctx.lineTo(right, quarterY);
        ctx.lineTo(right, threequarterY);
        ctx.lineTo(x, bottom);
        ctx.lineTo(left, threequarterY);
        ctx.lineTo(left, quarterY);
        ctx.lineTo(x, top);
        ctx.closePath();
        ctx.stroke();
        ctx.fill();
    }

    drawGrid(ctx, hl, viewport) {
        let w = hl.gridWidth;
        // assuming height will work with given width that takes canvas width
        let r = Math.floor(viewport.width / (w + .5) / 2);
        let xStep = 2 * r;
        let yStep = 1.5 * r;
        ctx.strokeStyle = "#000";
        for (let row = 0; row < hl.gridHeight; ++row) {
            let y = row * yStep + r;
            for (let col = 0; col < hl.gridWidth; ++col) {
                let x = col * xStep + r;
                x = (row % 2) === 0 ? x : x + r;
                ctx.fillStyle = HEX_CELL_COLORS[hl.getCell(col, row)];
                this.drawHex(ctx, x + viewport.x, y + viewport.y, r);
            }
        }
    }

    drawTransitionGrid(ctx, hl, oldHl, viewport) {
        let w = hl.gridWidth;
        // assuming height will work with given width that takes canvas width
        let r = Math.floor(viewport.width / (w + .5) / 2);
        let xStep = 2 * r;
        let yStep = 1.5 * r;
        ctx.strokeStyle = "#000";
        for (let row = 0; row < hl.gridHeight; ++row) {
            let y = row * yStep + r;
            for (let col = 0; col < hl.gridWidth; ++col) {
                let x = col * xStep + r;
                x = (row % 2) === 0 ? x : x + r;
                let colorSet1 = HEX_CELL_TRANSITIONS[ oldHl.getCell(col, row) ];
                let colorSet2 = HEX_CELL_TRANSITIONS[ hl.getCell(col, row) ];
                let red = Math.floor((colorSet2[0] - colorSet1[0]) * (1-this.transition) + colorSet1[0]);
                let green = Math.floor((colorSet2[1] - colorSet1[1]) * (1-this.transition) + colorSet1[1]);
                let blue = Math.floor((colorSet2[2] - colorSet1[2]) * (1-this.transition) + colorSet1[2]);
                let s  = `rgb(${red},${green},${blue})`;
//                console.log(s);
                ctx.fillStyle = s;
                this.drawHex(ctx, x + viewport.x, y + viewport.y, r);
            }
        }
    }

    render(ctx) {
        if (this.transition < 1) {
            this.transition = Math.min(1, this.transitionStep + this.transition);
        }
//        this.drawTransitionGrid(ctx, this.grid.workGrid, this.grid.gameGrid, this.bounds)
        this.drawGrid(ctx, this.grid.gameGrid, this.bounds)
    }

    mouseUp(x, y) {
        if (this.bounds.isInside(x, y))
            this.broadcast(this.getGridPoint(x,y));
    }
    getGridPoint(x, y) {
        let tx = x - this.bounds.x;
        let ty = y - this.bounds.y;
        let w = this.grid.gameGrid.gridWidth;
        let r = Math.floor(this.bounds.width / (w + .5) / 2);

        let step = Math.floor(r / 2);
        console.log(`getGridPoint(${x},${y}) r=${r} step=${step}`);
        let row = Math.floor((ty / step) / 3);
        let colAdj = (row % 2 === 1) ? -.5 : 0;
        let col = Math.floor(tx / (r * 2) + colAdj);
        return {"x": col, "y": row};
    }
}

// -------------------

class BasicScreen {
    componentList = [];
    manager;

    constructor(manager) {
        this.manager = manager;
    }

    buildButtonBar(firstBound, adjustBounds, labels) {
        let buttons = [];
        let x = firstBound.x;
        let y = firstBound.y;
        let w = firstBound.width;
        let h = firstBound.height;
        for(let s of labels) {
            if (s !== "") {
                let b = new Button(s, new Rect(x,y,w,h));
                buttons.push(b);
                b.addListener(this);
            }
            x += adjustBounds.x;
            y += adjustBounds.y;
            w += adjustBounds.width;
            h += adjustBounds.height;
        }
        return buttons;
    }

    mouseDown(x, y) {
        for (let c of this.componentList) {
            c.mouseDown(x, y);
        }
    }
    mouseMove(x, y) {
        for (let c of this.componentList) {
            c.mouseMove(x, y);
        }
    }
    mouseUp(x, y) {
        for (let c of this.componentList) {
            c.mouseUp(x, y);
        }
    }

    render(ctx) {
        for (let c of this.componentList) {
            c.render(ctx);
        }
    }

    guiMessage(component, message) {
        console.log("Unhandled GUI Message: " + message);
    }

    tick() { /* placeholder */ }
}

//

class HexLifeEditorScreen extends BasicScreen {
    currentEditCell = HEX_CELL_TYPES.RED;
    grid;
    gridGUI;
    editButtonLabels = ["Empty", "Red", "Green", "Blue", "Wall"];
    editButtonCells = [HEX_CELL_TYPES.EMPTY, HEX_CELL_TYPES.RED,
        HEX_CELL_TYPES.GREEN, HEX_CELL_TYPES.BLUE, HEX_CELL_TYPES.WALL];
    editButtons = [];
    testButton;
    clearButton;
    exitButton;

    constructor(manager) {
        super(manager);
        this.grid = new HexLifeGenerator(20,20);
        this.gridGUI = new HexLifeGUIGrid(this.grid,
            new Rect(10,10,610,450));
        this.gridGUI.addListener(this);
        this.componentList.push(this.gridGUI);

        let y = 64;
        for(let s of this.editButtonLabels) {
            let b = new Button(s, new Rect(620,y, 160,28))
            y += 32;
            b.addListener(this);
            this.componentList.push(b);
            this.editButtons.push(b);
        }

        y += 16;
        this.testButton = new Button("Test Map", new Rect(620,y, 160,28));
        this.testButton.addListener(this);
        this.componentList.push(this.testButton);

        y += 48;
        this.clearButton = new Button("Clear Map", new Rect(620,y, 160,28));
        this.clearButton.addListener(this);
        this.componentList.push(this.clearButton);

        y += 48;
        this.exitButton = new Button("Exit", new Rect(620,y, 160,28));
        this.exitButton.addListener(this);
        this.componentList.push(this.exitButton);

    }

    render(ctx) {
        super.render(ctx);
        ctx.save();
        ctx.fillStyle = HEX_CELL_COLORS[this.currentEditCell];
        ctx.fillRect(620,10, 160,40);
        ctx.restore();
    }

    guiMessage(component, message) {
        if (component === this.gridGUI) {
            console.log(`grid click @ ${message.x}, ${message.y}`)
            this.grid.gameGrid.setCell(message.x, message.y, this.currentEditCell);
        } else if (component === this.testButton) {
            let s = this.grid.gameGrid.toCondensedString();
            console.log(s);
            this.manager.runLevel(s);
        } else if (component === this.clearButton) {
            this.grid.gameGrid.clear();
        } else if (component === this.exitButton) {
            this.manager.popScreen();
        } else {
            for (let i = 0; i < this.editButtons.length; ++i) {
                let c = this.editButtons[i]
                if (c === component)
                    this.currentEditCell = this.editButtonCells[i];
            }
        }
    }
}

//
class HexLifeRunningScreen extends BasicScreen {
    gridGen = new HexLifeGenerator(20, 20);
    paused = false;

    runningCommands = [];
    pauseCommands = [];

    constructor(manager) {
        super(manager);
        this.gridGUI = new HexLifeGUIGrid(this.gridGen,
            new Rect(10,10,610,450));

        let startCommands = new Rect(620,20,160, 40);
        let AdjustCommands = new Rect(0,50,0, 0);

        this.runningCommands = this.buildButtonBar(
            startCommands, AdjustCommands, ["Pause", "", "Restart", "", "Exit"]);
        this.runningCommands.push(this.gridGUI);

        this.pauseCommands = this.buildButtonBar(
            startCommands, AdjustCommands, ["Start", "Step", "", "Restart", "", "Exit"]);
        this.pauseCommands.push(this.gridGUI);

        this.componentList = this.runningCommands;
    }

    guiMessage(component, message) {
        if (component === this.gridGUI)
            return;
        switch(component.text) {
            case "Pause":
                this.componentList = this.pauseCommands;
                this.paused = true;
                break;
            case "Start":
                this.componentList = this.runningCommands;
                this.paused = false;
                break;
            case "Step":
                this.gridGen.generateNextRound();
                break;
            case "Restart":
                this.gridGen.restoreSnapshot();
                break;
            case "Exit":
                this.manager.popScreen();
                break;
        }
    }

    setLevel(s) {
        this.gridGen.gameGrid.loadFromString(s);
        this.gridGen.createSnapshot();
    }

    tick() {
        if ( ( !this.paused) && (this.gridGUI.transition >= 1) ) {
            this.gridGen.generateNextRound();
            this.gridGUI.transition = 0;
        }
    }
}

// -------

class CampaignSetupScreen extends BasicScreen {
    gridGen = new HexLifeGenerator(20, 20);
    playerColor;
    playerMaxUsed;
    playerCurrentUsedCount;
    RemainingLabel = new Label("Remaining",
        new Rect(620,20, 160, 32));

    constructor(manager) {
        super(manager);
        this.componentList = this.buildButtonBar(
            new Rect(620,64, 160,28),
            new Rect(0,32,0,0),
            ["Start", "", "Reset", "", "Exit"]
        );
        this.gridGUI = new HexLifeGUIGrid(this.gridGen,
            new Rect(10,10,610,450));
        this.componentList.push(this.gridGUI);
        this.gridGUI.addListener(this);
        this.componentList.push(this.RemainingLabel);
    }

    setCampaignLevel(s, targetColor, canPlace) {
        this.gridGen.gameGrid.loadFromString(s);
        this.gridGen.createSnapshot();
        this.playerColor = targetColor;
        this.playerMaxUsed = canPlace;
        this.playerCurrentUsedCount = 0;
        this.RemainingLabel.text = "Remaining Cells: " + (this.playerMaxUsed-this.playerCurrentUsedCount);
        this.RemainingLabel.color = HEX_CELL_COLORS[targetColor];
    }

    guiMessage(component, message) {
        if (component === this.gridGUI) {
            console.log(`campaign map click ${message.x}, ${message.y}`);
            if (
                (this.gridGUI.grid.gameGrid.getCell(message.x, message.y) === HEX_CELL_TYPES.EMPTY) &&
                (this.playerCurrentUsedCount < this.playerMaxUsed)
            ) {
                this.gridGUI.grid.gameGrid.setCell(message.x, message.y, this.playerColor);
                ++this.playerCurrentUsedCount;
                this.RemainingLabel.text = "Remaining Cells: " + (this.playerMaxUsed-this.playerCurrentUsedCount);
            }
        } else {
            switch (component.text) {
                case "Reset":
                    this.gridGUI.grid.restoreSnapshot();
                    this.playerCurrentUsedCount = 0;
                    this.RemainingLabel.text = "Remaining Cells: " + (this.playerMaxUsed - this.playerCurrentUsedCount);
                    break;
                case "Start":
                    let s = this.gridGUI.grid.gameGrid.toCondensedString();
                    this.manager.runLevel(s);
                    break;
                case "Exit":
                    this.manager.popScreen();
                    break;
            }
        }
    }

}

// -----------------

class HexLifeTitleScreen extends BasicScreen {
    constructor(manager) {
        super(manager);
        this.componentList = this.buildButtonBar(
            new Rect(200, 250, 400,28),
            new Rect(0, 40, 0,0),
            ["Instructions", "Random Grid", "Editor", "Campaign"]
        )
        let titlePosition = new Rect(0,0,800,200);
        let titleLogo = new ImageLabel(
            TITLE_IMAGE, titlePosition, titlePosition);
        this.componentList.push(titleLogo);

    }
    guiMessage(component, message) {
        switch (component.text) {
            case "Instructions":
                this.manager.pushScreen(this.manager.instructionsScreen);
                break;
            case "Random Grid":
                let g = new HexLifeGenerator(20,20);
                g.generateRandomGrid(20,20,20);
                this.manager.runLevel(g.gameGrid.toCondensedString());
                break;
            case "Editor":
                this.manager.pushScreen(this.manager.editor);
                break;
            case "Campaign":
                this.manager.pushScreen(this.manager.campaignScreen);
                break;
        }
    }

}

// -----------------

class HexLifeCampaignScreen extends BasicScreen {
    constructor(manager) {
        super(manager);
        this.componentList = this.buildButtonBar(
            new Rect(200, 100, 400,28),
            new Rect(0, 40, 0,0),
            ["Make it Red 1",
                "Make it Red 2",
                "Make it Green 1",
                "Make it Green 2",
                "Make it Blue 1",
                "Make it Blue 2",
                "Exit"]
        )
        let campaignLabel = new Label(
            "Campaign Levels",  new Rect(0,10,800,100));
        campaignLabel.font = "64px Cursive";
        this.componentList.push(campaignLabel);

    }
    guiMessage(component, message) {
        switch (component.text) {
            case "Make it Red 1":
                this.manager.campaignPlayScreen.setCampaignLevel(
                    "....................\n"+
                    "....................\n"+
                    "....................\n"+
                    "....................\n"+
                    "....................\n"+
                    "....................\n"+
                    "....................\n"+
                    ".......BBBBB........\n"+
                    "....................\n"+
                    "....................\n"+
                    "....................\n"+
                    ".......GGGGG........\n"+
                    "....................\n"+
                    "....................\n"+
                    "....................\n"+
                    "....................\n"+
                    "....................\n"+
                    "....................\n"+
                    "....................\n"+
                    "....................\n", HEX_CELL_TYPES.RED, 5);
                this.manager.pushScreen(this.manager.campaignPlayScreen);
                break;

            case "Make it Red 2":
                this.manager.campaignPlayScreen.setCampaignLevel(
                    "....................\n" +
                    "....................\n" +
                    "B..B.BBB.B.B........\n" +
                    "B..B.B...BB.........\n" +
                    ".BBB.BBB..B.........\n" +
                    "GGGG.G....G.........\n" +
                    ".G.G.G....GG........\n" +
                    "G..G.GGG.G.G........\n" +
                    "....................\n" +
                    "....................\n" +
                    "....B..BB.BBB.BBB...\n" +
                    "...B...B..B...B.....\n" +
                    "....B...B..BB.BBB...\n" +
                    "...G...G..G...G.....\n" +
                    "....G...G..G..G.....\n" +
                    "...GGG.GG.G...GGG...\n" +
                    "....................\n" +
                    "....................\n" +
                    "....................\n" +
                    "...................." ,  HEX_CELL_TYPES.RED, 5);
                this.manager.pushScreen(this.manager.campaignPlayScreen);
                break;

            case "Make it Green 1":
                this.manager.campaignPlayScreen.setCampaignLevel(
                    "........BB..........\n" +
                    "........BB..........\n" +
                    "..*******BB*******..\n" +
                    "..*..............*..\n" +
                    "..*..............*..\n" +
                    "..*..**********..*..\n" +
                    "..*..*........*..*..\n" +
                    "..*..*........*..*..\n" +
                    "..*..*..*RR*..*..*..\n" +
                    "..*BBB..*RR*.....*..\n" +
                    "..*BBB..*RR*.....*..\n" +
                    "..*..*..*RR*..*..*..\n" +
                    "..*..*........*..*..\n" +
                    "..*..*........*..*..\n" +
                    "..*..**********..*..\n" +
                    "..*..............*..\n" +
                    "..*..............*..\n" +
                    "..*******RR*******..\n" +
                    ".........RR.........\n" +
                    "........RR.........." ,  HEX_CELL_TYPES.GREEN, 5);
                this.manager.pushScreen(this.manager.campaignPlayScreen);
                break;

            case "Make it Green 2":
                this.manager.campaignPlayScreen.setCampaignLevel(
                    "BB...*....RR....*..R\n" +
                    "B...*.....R.....*.RR\n" +
                    "B...*.........*..*..\n" +
                    "*..*..........*..*..\n" +
                    "*..*..*******..*....\n" +
                    "..*..*......*..*....\n" +
                    "..*..*.......*..*...\n" +
                    "....*..****..*..*...\n" +
                    "....*..*.RR*..*..*..\n" +
                    "...*..*..R*..*..*...\n" +
                    "...*..*..**..*..*...\n" +
                    "...*..*..*..*..*....\n" +
                    "....*..*....*..*....\n" +
                    "....*..*...*..*..*..\n" +
                    "..*..*..****..*..*..\n" +
                    "..*..*.......*..*...\n" +
                    "...*..*......*..*...\n" +
                    "...*..*******..*....\n" +
                    "BB..*..........*...B\n" +
                    "B...*.........*...BB" ,  HEX_CELL_TYPES.GREEN, 5);
                this.manager.pushScreen(this.manager.campaignPlayScreen);
                break;

            case "Make it Blue 1":
                this.manager.campaignPlayScreen.setCampaignLevel(
                    "GGG.......R.......RR\n" +
                    "GG........R.......RR\n" +
                    "GG........R........R\n" +
                    "..........R........R\n" +
                    "..........R.........\n" +
                    "..........R.........\n" +
                    "..........*.........\n" +
                    "..........*.........\n" +
                    "..........*.........\n" +
                    "GGGGGGG******RRRRRRR\n" +
                    "..........*.........\n" +
                    "..........*.........\n" +
                    "..........*.........\n" +
                    ".........G..........\n" +
                    "..........G.........\n" +
                    ".........G..........\n" +
                    "G........G..........\n" +
                    "G........G........RR\n" +
                    "GG.......G........RR\n" +
                    "GG.......G.......RRR\n" ,  HEX_CELL_TYPES.BLUE, 5);
                this.manager.pushScreen(this.manager.campaignPlayScreen);

                break;
            case "Make it Blue 2":
                this.manager.campaignPlayScreen.setCampaignLevel(
                    "....................\n" +
                    "....................\n" +
                    "..**..*****..*****..\n" +
                    "RR*..........*...*..\n" +
                    "RR*..............*..\n" +
                    "**..**..****...***..\n" +
                    "....*.......*..*GG..\n" +
                    "...*........*..*GG..\n" +
                    "..***..***...*..**..\n" +
                    "...*..*GGG..*..*....\n" +
                    "....*..*GGG.*..*....\n" +
                    "..**********..*..*..\n" +
                    "..*RRR........*..*..\n" +
                    "..*RRR.......*..*...\n" +
                    "..*****..*****..*...\n" +
                    "..*...*.............\n" +
                    "..*....*............\n" +
                    "..***..******..*****\n" +
                    "....................\n" +
                    "...................." ,  HEX_CELL_TYPES.BLUE, 5);
                this.manager.pushScreen(this.manager.campaignPlayScreen);
                break;

            case "Exit":
                this.manager.popScreen();
                break;
        }
    }

}

// -------------


class HexLifeInstructionsScreen extends BasicScreen {
    constructor(manager) {
        super(manager);
        this.componentList = this.buildButtonBar(
            new Rect(10, 400, 780,40),
            new Rect(0, 32, 0,0),
            ["Exit"]
        )
        let instructionsPosition = new Rect(0,0,800,400);
        let instructionsLogo = new ImageLabel(
            INSTRUCTIONS_IMAGE, instructionsPosition, instructionsPosition);
        this.componentList.push(instructionsLogo);

    }
    guiMessage(component, message) {
        switch (component.text) {
            case "Exit":
                this.manager.popScreen();
                break;
        }
    }

}

// ==============

class HexLifeGame {
    canvasElement;
    currentScreen;
    editor;
    campaignScreen;
    campaignPlayScreen;
    titleScreen;
    instructionsScreen;
    runner;
    screenStack = [];

    constructor(canvasId) {
        let c = document.getElementById(canvasId);
        this.canvasElement = c;
        c.addEventListener("mousedown", (event)=>{this.mouseDown(event)});
        c.addEventListener("mousemove", (event)=>{this.mouseMove(event)});
        c.addEventListener("mouseup", (event)=>{this.mouseUp(event)});

        this.titleScreen = new HexLifeTitleScreen(this);
        this.campaignScreen = new HexLifeCampaignScreen(this);
        this.campaignPlayScreen = new CampaignSetupScreen(this);
        this.runner = new HexLifeRunningScreen(this);
        this.editor = new HexLifeEditorScreen(this);
        this.instructionsScreen = new HexLifeInstructionsScreen(this);
        this.currentScreen = this.titleScreen;
    }

    draw() {
        let ctx = this.canvasElement.getContext('2d');

        ctx.fillStyle = "black";
        ctx.fillRect(0,0,800,450);
        this.currentScreen.render(ctx);
    }

    mouseDown(event) {
        let rect = event.target.getBoundingClientRect();
        let x = event.clientX - rect.x;
        let y = event.clientY - rect.y;
        this.currentScreen.mouseDown(x,y);
        //this.draw();
    }

    mouseMove(event) {
        let rect = event.target.getBoundingClientRect();
        let x = event.clientX - rect.x;
        let y = event.clientY - rect.y;
        this.currentScreen.mouseMove(x,y);
        //this.draw();
    }

    mouseUp(event) {
        let rect = event.target.getBoundingClientRect();
        let x = event.clientX - rect.x;
        let y = event.clientY - rect.y;
        this.currentScreen.mouseUp(x,y);
        //this.draw();
    }

    pushScreen(newScreen) {
        this.screenStack.push(this.currentScreen);
        this.currentScreen = newScreen;
    }

    popScreen() {
        if (this.screenStack.length >= 1)
            this.currentScreen = this.screenStack.pop();
        else
            this.currentScreen = this.titleScreen;
    }

    runLevel(s) {
        this.runner.setLevel(s);
        this.pushScreen(this.runner);
    }
    tick() {
        this.currentScreen.tick();
        this.draw();
    }
}
